;;; cdr255-numeric.el --- Numeric Library   -*- lexical-binding: t; -*-

;; Copyright (C) 2025 Claire Rodriguez

;; Author: Claire Rodriguez <yewscion@gmail.com>
;; Keywords: numbers
;; Version: 0.0.5

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; 

;;; Code:

(require 'cdr255-textual)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;  Level 0: Primitives Only  ;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-numeric:whitaker-char-p (char)
  "Predicate for detecting if CHAR is a Whitaker Hexadecimal
Digit. Currently, those are the following characters:



This is a CALCULATION. This is a predicate.

Arguments
=========

CHAR<char>: The character to test.

Returns
=======

A <boolean>: <t> if CHAR is a Whitaker Hexadecimal Digit. <nil>
otherwise.

Impurities
==========

None."
  (and (<= char ?\xecff) (>= char ?\xecf0)))

(defun cdr255-numeric:hex-char-p (char)
  "A predicate for testing if CHAR is a valid naive hexadecimal
digit. Currently those are:

0123456789ABCDEFabcdef

This is a CALCULATION. This is a predicate.

Arguments
=========

CHAR<char>: The character to test.

Returns
=======

A <boolean>: <t> if CHAR is a valid hexadecimal digit character; <nil>
otherwise.

Impurities
==========

None."
  (or
   (and (>= char ?\x30) (<= char ?\x3a))
   (and (>= char ?\x41) (<= char ?\x46))
   (and (>= char ?\x61) (<= char ?\x66))))

(defun cdr255-numeric:binary-char-p (char)
  "A predicate for testing if CHAR is a valid binary digit
character. Currently those are:

01

This is a CALCULATION. This is a predicate.

Arguments
=========

CHAR<char>: A character to test.

Returns
=======

A <boolean>: <t> if CHAR is a valid binary digit character; <nil>
otherwise.

Impurities
==========

None."
  (or (= char ?0) (= char ?1)))

(defun cdr255-numeric:place-values→numeric (place-values
                                             base
                                             &optional
                                             acc
                                             rank)
  "Convert a given list of PLACE-VALUES into a numeric value based on a
given numeric BASE.

This is a CALCULATION.

Arguments
=========

PLACE-VALUES<<list> of <numbers>>: The rank values of a number in a
                                   given base, denoted using numeric
                                   values, in ascending order.


ACC<number>: For recursion; the accumulated numeric value of the
             computation so far.

RANK<number>: For recursion; the current rank to work on in the
              computation.

Returns
=======

A <number> representing the numeric value of the given PLACE-VALUES in
the given BASE.

Impurities
==========

None."
  (let ((acc (if acc acc 0))
        (rank (if rank rank 1)))
    (cond ((< (length place-values) 1)
           acc)
          (t
           (cdr255-numeric:place-values→numeric
            (cdr place-values)
            base
            (+ acc (* (car place-values)
                      (expt base (- rank 1))))
            (+ rank 1))))))

(defun cdr255-numeric:highest-rank-needed (number base &optional acc)
  "Calculate the highest rank needed to represent a given NUMBER in a given
BASE.

This is a CALCULATION.

Arguments
=========

NUMBER<number>: The number for which conversion is desired.

BASE<number>: The base to which we want to convert NUMBER.

ACC<number>: For recursion; The current count of needed ranks.

Returns
=======

A <number> representing the highest rank needed to represent NUMBER in
the given BASE.

Impurities
==========

None."
  (let ((acc (if acc acc 0)))
    (if (> (expt base acc) number)
        (- acc 1)
      (cdr255-numeric:highest-rank-needed number base (+ acc 1)))))

(defun cdr255-numeric:natural-number-p (number)
  "Predicate to test if a given NUMBER is in ℕ (natural numbers including
0).

This is a CALCULATION.

Arguments
=========

NUMBER<number>: The number to test.

Returns
=======

A <boolean>: <t> if NUMBER is in ℕ, nil otherwise.

Impurities
==========

None."
  (and (integer-or-marker-p number) (> number -1)))

(defun cdr255-numeric:place-values→string (place-values
                                           conversion-proc)
  "Convert the given list of PLACE-VALUES into a string, using
CONVERSION-PROC to convert the numbers into the symbols to represent
them.

This is a CALCULATION.

Arguments
=========

PLACE-VALUES<<list> of <numbers>>: An ascending-order place-value list
                                   for a specific numeric quantity.

CONVERSION-PROC<procedure>: A procedure that takes a single <number> as a
                            parameter and returns a <character>
                            representing a glyph used to represent that
                            <number>.

Returns
=======

A <string> representing the given PLACE-VALUES as interpreted by the CONVERSION-PROC.

Impurities
==========

None."
  (apply #'string (reverse (mapcar conversion-proc place-values))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;  Level 1: Library Primitives Only  ;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-numeric:number-string→place-values (number-string
                                                   conversion-proc)
  "Convert a given NUMBER-STRING digit-by-digit using a given
CONVERSION-PROC -edure into an ascending-order place-value list using
numeric values. Example: `1024' would become `(4 2 0 1)'.

This is a CALCULATION.

Arguments
=========

NUMBER-STRING<string>: A sequence of digit characters in a numeric base
                       with a ...→numeric conversion procedure defined.

CONVERSION-PROC<procedure>: A procedure that takes a character and
                            produces the numeric value denoted by it in
                            a specific notation scheme.

Returns
=======

A <<list> of <numbers>> denoting the ascending ranked numeric value of
the original NUMBER string. NOTE: Does *not* record a base, so it is
possible to lose track of the referred-to value if used for storage.

Impurities
==========

None."
  (reverse
   (cdr255:convert-string-as-characters number-string conversion-proc)))

(defun cdr255-numeric:ensure-lowercase-hex-char (char)
  "Coerce CHAR into a /lowercase/ hexadecimal digit.

This is a CALCULATION.

Arguments
=========

CHAR<char>: A (hopefully) hexadecimal digit character. Non-hexadecimal
characters are an error.

Returns
=======

A <char> representing the original CHAR, but downcased if the original
was uppercase.

Impurities
==========

None."
  (let ((char (cdr255-textual:ensure-char-or-single-char-string char)))
    (if (cdr255-numeric:hex-char-p
         char)
        (if (and (>= char ?\x41) (<= char ?\x46))
            (+ char 32)
          char)
      (error "'%c' is not a hexadecimal digit!" char))))

(defun cdr255-numeric:ensure-binary-char (char)
  "Coerce CHAR into a binary digit.

This is a CALCULATION.

Arguments
=========

CHAR<char>: A (hopefully) binary digit character. Non-binary digit
characters are an error.

Returns
=======

A <char> representing the original CHAR.

Impurities
==========

None."
  (let ((char (cdr255-textual:ensure-char-or-single-char-string char)))
    (if (cdr255-numeric:binary-char-p
         char)
        char
      (error "'%c' is not a binary digit!" char))))

(defun cdr255-numeric:numeric→place-values (number base &optional rank)
  "Convert a given NUMBER into an ascending-order place-value list using
numeric values. Example: `1024' would become `(4 2 0 1)'.

This is a CALCULATION.

Arguments
=========

NUMBER<number>: The number to convert.

BASE<number>: The numeric base of which each rank is considered to be a
part.

RANK<number>: For recursion; The current rank being calculated.

Returns
=======

A <<list> of <numbers>> representing an ascending-order place-value list
using numeric values. Example: `1024' would become `(4 2 0 1)'.

Impurities
==========

None."
  (let ((rank (if rank rank (cdr255-numeric:highest-rank-needed number
                                                                base))))
    (cond ((not (cdr255-numeric:natural-number-p number))
           (error "Negative and fractional numbers are a work in progress
still."))
          ((= rank 0)
           (list number))
          (t
           (append (cdr255-numeric:numeric→place-values
                    (mod number (expt base rank))
                    base
                    (- rank 1))
                 (list (/ number (expt base rank))))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;  To Numeric Value  ;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun cdr255-numeric:whitaker-char→numeric (whitaker)
  "Convert WHITAKER into its numeric representation.

This is a CALCULATION.

Arguments
=========

WHITAKER<char>: A Whitaker Hexadecimal Digit character. Currently, those
                are the following characters:

                

Returns
=======

A <number> representing the numeric value of the original WHITAKER
character.

Impurities
==========

None."

  (let ((whitaker
         (cdr255-textual:ensure-char-or-single-char-string
          whitaker)))
    (if (cdr255-numeric:whitaker-char-p whitaker)
        (- whitaker ?\xecf0)
      (error "`%c' is not a Whitaker Hexidecimal Digit." whitaker))))

(defun cdr255-numeric:hex-char→numeric (hex-char)
  "Convert HEX-CHAR into its numeric representation.

This is a CALCULATION.

Arguments
=========

HEX-CHAR<char>: A naive hexadecimal digit character. Currently, those
                are the following characters:

                0123456789ABCDEFabcdef

Returns
=======

A <number> representing the numeric value of the original HEX-CHAR.

Impurities
==========

None."

  (let ((hex-char (cdr255-numeric:ensure-lowercase-hex-char
                   hex-char)))
    (if (and (>= hex-char ?\x30) (<= hex-char ?\x3a))
        (- hex-char ?\x30)
      (- hex-char ?\x57))))

(defun cdr255-numeric:binary-char→numeric (binary-char)
  "Convert BINARY-CHAR into its numeric representation.

This is a CALCULATION.

Arguments
=========

BINARY-CHAR<char>: A valid binary digit character. Currently those are:

                   01

Returns
=======

A <number> representing the numeric value of the original BINARY-CHAR.

Impurities
==========

None."
  (let ((binary-char (cdr255-numeric:ensure-binary-char
                      binary-char)))
    (- binary-char 48)))

(defun cdr255-numeric:numeric→character-list (numeric
                                               character-list
                                               character-set-name)
  (if (and (< numeric (length character-list))
           (cdr255-numeric:natural-number-p
            numeric))
      (nth numeric character-list)
    (error "`%d' is not expressable as a single %s digit."
           numeric
           character-set-name)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;  Level 2: Compound Procedures  ;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-numeric:numeric→whitaker-char (numeric)
  "Convert the given NUMERIC value into the appropriate Whitaker
Hexidecimal character.

This is a CALCULATION.

Arguments
=========

NUMERIC<number>: A number x such that 0 ≯ x ≯ 16.

Returns
=======

A <character> representing the representation of the given numeric value
using a single Whitaker Hexadecimal digit.

Impurities
==========

None."
  (cdr255-numeric:numeric→character-list
   numeric
   '(? ? ? ? ? ? ? ? ? ? ? ? ? ? ? ?)
   "whitaker"))

(defun cdr255-numeric:numeric→hex-char (numeric)
  "Convert the given NUMERIC value into the appropriate Conventional
Hexidecimal character.

This is a CALCULATION.

Arguments
=========

NUMERIC<number>: A number x such that 0 ≯ x ≯ 16.

Returns
=======

A <character> representing the representation of the given numeric value
using a single Conventional Hexadecimal digit.

Impurities
==========

None."
  (cdr255-numeric:numeric→character-list
   numeric
   '(?0 ?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?a ?b ?c ?d ?e ?f)
   "hexadecimal"))

(defun cdr255-numeric:numeric→binary-char (numeric)
  "Convert the given NUMERIC value into the appropriate Binary
character.

This is a CALCULATION.

Arguments
=========

NUMERIC<number>: A number x such that 0 ≯ x ≯ 1

Returns
=======

A <character> representing the representation of the given numeric value
using a single Binary digit.

Impurities
==========

None."
  (cdr255-numeric:numeric→character-list
   numeric
   '(?0 ?1)
   "binary"))

(defun cdr255-numeric:number-string→numeric (number-string
                                              conversion-proc
                                              base)
  "Convert a given NUMBER-STRING into a specific BASE, using the given
CONVERSION-PROCEDURE digit-by-digit, into a numeric value.

This is a CALCULATION.

Arguments
=========

NUMBER-STRING<string>: A sequence of digit characters in a numeric base
                       with a ...→numeric conversion procedure defined.

CONVERSION-PROC<procedure>: A procedure that takes a character and
                            produces the numeric value denoted by it in
                            a specific notation scheme.

BASE<number>: The numeric base in which to interpret the given
              NUMBER-STRING.

Returns
=======

A <number> representing the numeric value of the given NUMBER-STRING,
interpreted using the given BASE.

Impurities
==========

None."
  (let ((place-values
         (cdr255-numeric:number-string→place-values
          number-string
          conversion-proc)))
    (cdr255-numeric:place-values→numeric place-values base)))

(defun cdr255-numeric:binary-string→numeric (binary-string)
  "Convert a given BINARY-STRING (That is, a string of binary digits) into
its numeric form.

This is a CALCULATION.

Arguments
=========

BINARY-STRING<string>: The string representing a number in binary
                       notation.

Returns
=======

A <number> representing the numeric value denoted by the given
BINARY-STRING.

Impurities
==========

None."
  (cdr255-numeric:number-string→numeric
   binary-string
   #'cdr255-numeric:binary-char→numeric
   2))

(defun cdr255-numeric:hex-string→numeric (hex-string)
  "Convert a given HEX-STRING (That is, a string of hexadecimal digits) into
its numeric form.

This is a CALCULATION.

Arguments
=========

HEX-STRING<string>: The string representing a number in hexadecimal
                       notation.

Returns
=======

A <number> representing the numeric value denoted by the given
HEX-STRING.

Impurities
==========

None."
  (cdr255-numeric:number-string→numeric
   hex-string
   #'cdr255-numeric:hex-char→numeric
   16))

(defun cdr255-numeric:whitaker-string→numeric (whitaker-string)
  "Convert a given WHITAKER-STRING (That is, a string of whitaker
hexadecimal digits) into its numeric form.

This is a CALCULATION.

Arguments
=========

WHITAKER-STRING<string>: The string representing a number in whitaker
                         hexadecimal notation.

Returns
=======

A <number> representing the numeric value denoted by the given
WHITAKER-STRING.

Impurities
==========

None."
  (cdr255-numeric:number-string→numeric
   whitaker-string
   #'cdr255-numeric:whitaker-char→numeric
   16))



;;;;;;;;;;;;;;;;;;;;;;;;;;  From Numeric Value  ;;;;;;;;;;;;;;;;;;;;;;;;;;
 
(defun cdr255-numeric:place-values→whitaker-string (place-values)
  "Convert the given list of PLACE-VALUES into a Whitaker Hexadecimal
string.

This is a CALCULATION.

Arguments
=========

PLACE-VALUES<<list> of <numbers>>: An ascending-order place-value list
                                   for a specific numeric quantity.

Returns
=======

A <string> representing the given PLACE-VALUES as a Whitaker Hexadecimal
String.

Impurities
==========

None."  
  (cdr255-numeric:place-values→string
   place-values
   #'cdr255-numeric:numeric→whitaker-char))
(defun cdr255-numeric:place-values→hex-string (place-values)
  "Convert the given list of PLACE-VALUES into a Conventional Hexadecimal
string.

This is a CALCULATION.

Arguments
=========

PLACE-VALUES<<list> of <numbers>>: An ascending-order place-value list
                                   for a specific numeric quantity.

Returns
=======

A <string> representing the given PLACE-VALUES as a Conventional Hexadecimal
String.

Impurities
==========

None."
  (cdr255-numeric:place-values→string
   place-values
   #'cdr255-numeric:numeric→hex-char))
(defun cdr255-numeric:place-values→binary-string (place-values)
  "Convert the given list of PLACE-VALUES into a Binary
string.

This is a CALCULATION.

Arguments
=========

PLACE-VALUES<<list> of <numbers>>: An ascending-order place-value list
                                   for a specific numeric quantity.

Returns
=======

A <string> representing the given PLACE-VALUES as a Binary
String.

Impurities
==========

None."
  (cdr255-numeric:place-values→string
   place-values
   #'cdr255-numeric:numeric→binary-char))

;;;;;;;;;;;;;;;;;;;;;;;;;  Through Numeric Value  ;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-numeric:hex-string→whitaker-string (hex-string)
  "Convert a Conventional Hexadecimal string into a Whitaker Hexadecimal
string representing the same number.

This is a CALCULATION.

Arguments
=========

HEX-STRING<string>: A string composed entirely of Conventional Hexadecimal digits.

Returns
=======

A <string> representing the same numeric quantity as HEX-STRING, but
using Whitaker Hexadecimal Notation.

Impurities
==========

None."
  (cdr255-numeric:place-values→whitaker-string
   (cdr255-numeric:numeric→place-values
    (cdr255-numeric:hex-string→numeric hex-string)
    16)))

(defun cdr255-numeric:whitaker-string→hex-string (whitaker-string)
  "Convert a Whitaker Hexadecimal string into a Conventional Hexadecimal
string representing the same number.

This is a CALCULATION.

Arguments
=========

HEX-STRING<string>: A string composed entirely of Whitaker Hexadecimal digits.

Returns
=======

A <string> representing the same numeric quantity as HEX-STRING, but
using Conventional Hexadecimal Notation.

Impurities
==========

None."
  (cdr255-numeric:place-values→hex-string
   (cdr255-numeric:numeric→place-values
    (cdr255-numeric:whitaker-string→numeric whitaker-string)
    16)))

(defun cdr255-numeric:whitaker-string→binary-string (whitaker-string)
  "Convert a Whitaker Hexadecimal string into a Binary
string representing the same number.

This is a CALCULATION.

Arguments
=========

HEX-STRING<string>: A string composed entirely of Whitaker Hexadecimal digits.

Returns
=======

A <string> representing the same numeric quantity as HEX-STRING, but
using Binary Notation.

Impurities
==========

None."
  (cdr255-numeric:place-values→binary-string
   (cdr255-numeric:numeric→place-values
    (cdr255-numeric:whitaker-string→numeric whitaker-string)
    2)))

(defun cdr255-numeric:binary-string→whitaker-string (binary-string)
  "Convert a Binary string into a Whitaker Hexadecimal
string representing the same number.

This is a CALCULATION.

Arguments
=========

HEX-STRING<string>: A string composed entirely of Binary digits.

Returns
=======

A <string> representing the same numeric quantity as HEX-STRING, but
using Whitaker Hexadecimal Notation.

Impurities
==========

None."
  (cdr255-numeric:place-values→whitaker-string
   (cdr255-numeric:numeric→place-values
    (cdr255-numeric:binary-string→numeric binary-string)
    16)))

(defun cdr255-numeric:hex-string→binary-string (hex-string)
  "Convert a Conventional Hexadecimal string into a Binary
string representing the same number.

This is a CALCULATION.

Arguments
=========

HEX-STRING<string>: A string composed entirely of Conventional Hexadecimal digits.

Returns
=======

A <string> representing the same numeric quantity as HEX-STRING, but
using Binary Notation.

Impurities
==========

None."
  (cdr255-numeric:place-values→binary-string
   (cdr255-numeric:numeric→place-values
    (cdr255-numeric:hex-string→numeric hex-string)
    2)))

(defun cdr255-numeric:binary-string→hex-string (binary-string)
  "Convert a Binary string into a Conventional Hexadecimal
string representing the same number.

This is a CALCULATION.

Arguments
=========

HEX-STRING<string>: A string composed entirely of Binary digits.

Returns
=======

A <string> representing the same numeric quantity as HEX-STRING, but
using Conventional Hexadecimal Notation.

Impurities
==========

None."
  (cdr255-numeric:place-values→hex-string
   (cdr255-numeric:numeric→place-values
    (cdr255-numeric:binary-string→numeric binary-string)
    16)))

(defun cdr255-numeric:detect-numeric-character-set (str)
  (cond ((cdr255-textual:check-string-against-character-list-p
          str
          '(?1 ?0))
         'binary)
        ((cdr255-textual:check-string-against-character-list-p
          str
          '(?0 ?1 ?2 ?3 ?4 ?5 ?6 ?7 ?8 ?9 ?a ?b ?c ?d ?e ?f))
         'hexadecimal)
        ((cdr255-textual:check-string-against-character-list-p
          str
          '(?\xecf0 ?\xecf1 ?\xecf2 ?\xecf3 ?\xecf4 ?\xecf5 ?\xecf6 ?\xecf7
                    ?\xecf8 ?\xecf9 ?\xecfa ?\xecfb ?\xecfc ?\xecfd ?\xecfe
                    ?\xecff))
         'whitaker)
        (t
         (error "Unable to detect which numeric character set %s uses! See %s"
                str "cdr255-numeric:detect-numeric-character-set"))))

(defun cdr255-numeric:region-to-whitaker (start end &optional lst)
  (let* ((str (buffer-substring-no-properties start end))
         (type (cdr255-numeric:detect-numeric-character-set
                str)))
    (cond ((eq type 'binary)
           (cdr255-numeric:binary-string→whitaker-string str))
          ((eq type 'hexadecimal)
           (cdr255-numeric:hex-string→whitaker-string str))
          ((eq type 'whitaker)
           str)
          (t
           (error "Unable to convert a %s type number to whitaker, see %s."
                  type
                  "cdr255-numeric:region-to-whitaker")))))

(defun cdr255-numeric:region-to-hexadecimal (start end &optional lst)
  (let* ((str (buffer-substring-no-properties start end))
         (type (cdr255-numeric:detect-numeric-character-set
                str)))
    (cond ((eq type 'binary)
           (cdr255-numeric:binary-string→hex-string str))
          ((eq type 'hexadecimal)
           str)
          ((eq type 'whitaker)
           (cdr255-numeric:whitaker-string→hex-string str))
          (t
           (error "Unable to convert a %s type number to hexadecimal, see %s."
                  type
                  "cdr255-numeric:region-to-hexadecimal")))))

(defun cdr255-numeric:region-to-binary (start end &optional lst)
  (let* ((str (buffer-substring-no-properties start end))
         (type (cdr255-numeric:detect-numeric-character-set
                str)))
    (cond ((eq type 'binary)
           str)
          ((eq type 'hexadecimal)
           (cdr255-numeric:hex-string→binary-string str))
          ((eq type 'whitaker)
           (cdr255-numeric:whitaker-string→binary-string str))
          (t
           (error "Unable to convert a %s type number to binary, see %s."
                  type
                  "cdr255-numeric:region-to-binary")))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;  Level 3: Commands  ;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-numeric:convert-to-whitaker ()
  (interactive)
  (cdr255:replace-region #'cdr255-numeric:region-to-whitaker))

(defun cdr255-numeric:convert-to-hexadecimal ()
  (interactive)
  (cdr255:replace-region #'cdr255-numeric:region-to-hexadecimal))

(defun cdr255-numeric:convert-to-binary ()
  (interactive)
  (cdr255:replace-region #'cdr255-numeric:region-to-binary))

(provide 'cdr255-numeric)
;;; cdr255.el ends here
