;;; cdr255-textual.el --- Numeric Library   -*- lexical-binding: t; -*-

;; Copyright (C) 2025 Claire Rodriguez

;; Author: Claire Rodriguez <yewscion@gmail.com>
;; Keywords: text
;; Version: 0.0.5

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; 

;;; Code:

(require 'cdr255)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;  Level 0: Primitives Only  ;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-textual:ensure-char-or-single-char-string (char
                                                         &optional
                                                         msg)
  "Coerce CHAR into a single char, with an optional error message if it
is impossible to do so.

This is a CALCULATION. This is a guard.

Arguments
=========

CHAR<<char> or <string>>: (Hopefully) A char or a string of length 1. A
multi-character string is an error.

MSG<string>: An error message to use when a multi-character string is
detected.

Returns
=======

A <char> representing the original CHAR.

Impurities
==========

None."
  (let ((msg (if msg msg "`%s' is not a single char string!")))
    (if (stringp char)
        (if (> (length char) 1)
            (error msg char)
          (string-to-char char))
      char)))


(defun cdr255-textual:replace-regexp-iteratively-in-string
    (str regexp-alist)
  "For each member of REGEXP-ALIST, relace all matches of the regexp (in
the car) of STR with the given string (in the cdr).

Operations are done /in sequence/: The first entry is done first, and so
on. Organize REGEXP-ALIST accordingly.

This is a CALCULATION.

Arguments
=========

STR<string>: The original string to operate on.

REGEXP-ALIST<<list> of <pairs>>: A list of pairs, with the car being a
regexp string and the cdr being a string to replace it with, in the
order they are to be applied to the STR.

Returns
=======

A <string> representing STR with all of the replacements defined in
REGEXP-ALIST having been applied in order to it.

Impurities
==========

None."
  (cond ((= (list-length regexp-alist) 1)
         (replace-regexp-in-string (car (car regexp-alist))
                                   (cdr (car regexp-alist))
                                   str
                                   t))
        (t
         (cdr255-textual:replace-regexp-iteratively-in-string
          (replace-regexp-in-string (car (car regexp-alist))
                                    (cdr (car regexp-alist))
                                    str
                                    t)
          (cdr regexp-alist)))))

(defun cdr255-textual:link-alist-to-org-link (link-alist)
  "Convert LINK-ALIST into an Org-style link.

This is a CALCULATION.

Arguments
=========

LINK-ALIST<association-list>: a `Link Alist', with the keys `url' and
                              `text' containing the appropriate values.

Returns
=======

A <string> representing the link defined in LINK-ALIST as an Org-style
link (that is, `[[url][text]]'.

Impurities
==========

None."
  (concat "[["
          (alist-get 'url link-alist)
          "]["
          (alist-get 'text link-alist)
          "]]"))

(defun cdr255-textual:string-to-set-of-characters (str)
  "Represent all unique characters in STR in a list.

This is a CALCULATION.

Arguments
=========

STR<string>: The original string.

Returns
=======

A <<list> of <characters>> with one member for every unique character in
STR.

Impurities
==========
None."
  (cl-remove-duplicates (string-to-list str)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;  Level 1: Library Primitives Only  ;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-textual:md-link-to-alist (str)
  "Convert STR (which should be a Markdown-style link) into an alist
suitable for the cdr255-textual:link-alist-to-* procedures.

This is a CALCULATION.

Arguments
=========

STR<string>: Assumed to be a Markdown-style link (that is,
             `[display text](url)').

Returns
=======

A <association list> representing STR as a `Link Alist', with the keys
`url' and `text' containing the appropriate values.

Impurities
==========

None."
  (let ((lst (string-split
             (cdr255-textual:replace-regexp-iteratively-in-string
              str '(("[]()[]" . "")("+" . "")("^\\|$" . "")))
             "")))
    `((url . ,(cadr lst))
      (text . ,(car lst)))))

(defun cdr255-textual:check-string-against-character-list-p
    (str character-list)
  "Check whether STR is solely composed of characters in CHARACTER-LIST.

This is a CALCULATION. This is a predicate.

Arguments
=========

STR<string>: The string to test.

CHARACTER-LIST<<list> of <characters>>: A list of characters which is
                                        treated like a defined set to
                                        check against.

Returns
=======

A <boolean>: <true> if STR only contains characters that are present in
CHARACTER-LIST. <false> otherwise.

Impurities
==========

None."
  (reduce #'cdr255:and (mapcar
                        (lambda (x) (cdr255:to-boolean (member x character-list)))
                        (cdr255-textual:string-to-set-of-characters
                         str))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;  Level 2: Compound Procedures  ;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-textual:parse-link-to-alist (str)
  "Convert STR, which is some kind of hyperlink represented as a text
string, into a `Link Alist'.

This is a Calculation. This is a Dispatcher.

Arguments
=========

STR<string>: The string with should be some kind of marked-up hyperlink.

Returns
=======

A <association list> representing STR as a `Link Alist', with the keys
`url' and `text' containing the appropriate values.

Impurities
==========

None."

  (cond ((and (string= (substring str 0 1) "[")
              (string= (substring str -1) ")"))
         (cdr255-textual:md-link-to-alist str))
        (t
         str)))

(defun cdr255-textual:md-link→org-link (start end &optional lst)
  "Convert a region of text defined as starting at START and ending at END
from a Markdown-style link into an Org-style link.

This is an ACTION.

Arguments
=========

START<number>: The position in the current buffer at which to start the
               region.

END<number>: The position in the current buffer at which to end the
             region.

LST<nil>: Ignored, for interfacing with `cdr255::operate-on-region'.

Returns
=======

A <string> representing the buffer-substring starting at START and
ending at END being interpreted as a Markdown-style link and converted
into an Org-style link.

Impurities
==========

Interfaces with current buffer."

  (let ((original (buffer-substring-no-properties start end)))
    (delete-region start end)
    (insert (cdr255-textual:link-alist-to-org-link
             (cdr255-textual:md-link-to-alist original)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;  Level 3: Commands  ;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255-textual:convert-md-link-to-org-link ()
  "Replaces the Markdown-style link in the region with an Org-style link.

This is an ACTION.

Arguments
=========

<none>

Returns
=======

<undefined>

Impurities
==========

Used solely for side effects."
  (interactive)
  (cdr255:operate-on-region #'cdr255-textual:md-link→org-link))

(provide 'cdr255-textual)
