;;; cdr255-substitutions.el --- Yewscion's Substitution Library  -*- lexical-binding: t; -*-

;; Copyright (C) 2025 Claire Rodriguez

;; Author: Claire Rodriguez <yewscion@gmail.com>
;; Keywords: abbrev
;; Version: 0.0.3

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; 

;;; Code:

(defmacro cdr255-substitutions:define-substitution
    (original-sequence result)
  "Define a sequence usable to substitute harder-to-type characters
in, using the substitution-map."
  `(define-key substitution-map (kbd ,original-sequence)
               #'(lambda nil (interactive) (insert ,result))))

(defmacro cdr255-substitutions:define-substitution-prefix
    (name key)
  "Define a prefix key for use with substitutions, as I have to
 many not to me utilizing prefixes."
  `(progn
     (define-prefix-command (intern (concat "substitution-"
                                            ,name
                                            "-prefix-map")))
     (bind-key (kbd ,key) (intern (concat "substitution-"
                                          ,name
                                          "-prefix-map"))
               substitution-map)))

(defmacro cdr255-substitutions:define-prefixed-substitution
    (prefix original result)
  "Define a substitution under a specific prefix."
  `(bind-key (kbd ,original) (kbd ,result)
             (intern (concat "substitution-"
                             ,prefix
                             "-prefix-map"))))

(defun cdr255-substitutions:define-substitution-grouping
    (name key alist)
  "Creates substitution-NAME-prefix-map, assigns it to KEY under
the substitution-map, and then assigns each member of the ALIST
as substitutions in that map, in the format of (ORIGINAL-SEQUENCE
. SUBSTITUTION). Then creates procedures to display the keymap on
a textual version of various keyboards."
  (cdr255-substitutions:define-substitution-prefix name key)
  (mapc (lambda (x)
          (cdr255-substitutions:define-prefixed-substitution name
                                                             (car x)
                                                             (cdr x)))
        alist)
  (cdr255:eval-string
   (cdr255-substitutions:generate-string-for-substitution-map-display-proc
    "macbook-air"
    name )))

(defmacro cdr255-substitutions:display-macbook-air-keymap (keymap)
  `(cdr255:temp-buffer-display-string
    (symbol-name (quote ,keymap))
    (cdr255-substitutions:generate-macbook-air-m2-keyboard ,keymap)))

(defun cdr255-substitutions:generate-string-for-substitution-map-display-proc
    (type name)
  (format
   "(defun cdr255-substitutions:display-%s-substitution-%s-prefix-map-keyboard ()
\"Displays the keymap defined by substitution-%s-prefix-map as a %s
keyboard diagram.\"
(interactive)
(cdr255-substitutions:display-%s-keymap substitution-%s-prefix-map))"
   type
   name
   name
   type
   type
   name))

(defmacro cdr255-substitutions:dereference-key-in-keymap
    (key keymap)
  `(let ((sequence (keymap-lookup ,keymap ,key)))
     (if sequence
         (key-description sequence)
       ,key)))

(defmacro cdr255-substitutions:dereference-key-in-keymap-with-default
    (key keymap default)
  `(let ((sequence (keymap-lookup ,keymap ,key)))
     (if sequence
         (key-description sequence)
       ,default)))

(defun cdr255-substitutions:get-key-from-keymap-at-2-columns (key
                                                              keymap
                                                              &optional
                                                              default)
  (let ((result
         (if default
             (cdr255-substitutions:dereference-key-in-keymap-with-default
              key
              keymap
              default)
           (cdr255-substitutions:dereference-key-in-keymap key keymap))))
    (if (< (string-pixel-width result) (string-pixel-width "  "))
        (if (< (string-pixel-width (concat result " "))
               (string-pixel-width "  "))
            (concat result "  ")
          (concat result " "))
      result)))

(defun cdr255-substitutions:generate-macbook-air-m2-keyboard (keymap)
  (format
   " ┌───────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬─────┐ 
 │ %s    │ 🔅 │ 🔆 │ 🗍 │ 🔍 │ 🎙 │ 🌙 │ ⏪ │ ⏯ │ ⏩ │ 🔈 │ 🔉 │ 🔊 │ ⏻  │ 
 │       │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │     │ 
 └───────┴────┴────┴────┴────┴────┴────┴────┴────┴────┴────┴────┴────┴─────┘
 ┌────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬────┬────────┐ 
 │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │        │ 
 │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s     │ 
 ├────┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─────┤ 
 │  %s   │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s  │ 
 │  %s   │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s  │ 
 ├───────┴┬───┴┬───┴┬───┴┬───┴┬───┴┬───┴┬───┴┬───┴┬───┴┬───┴┬───┴┬───┴─────┤ 
 │ ^Ctrl  │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │         │ 
 │        │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s      │ 
 ├────────┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴──┬─┴─────────┤ 
 │           │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │           │ 
 │ ⇧ Shift   │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │ %s │   %s      │ 
 ├────┬─────┬┴───┬┴────┼────┴────┴────┴────┴──┬─┴───┬┴───┬┴────┴───────────┘ 
 │  fn│     │    │     │           %s         │     │    │       ┌────┐
 │🌐  │ ⑮  │ ⌘  │ ⌥   │                      │ %s  │ %s │       │ %s │
 └────┴─────┴────┴─────┴──────────────────────┴─────┴────┘  ┌────┼────┼────┐
                                                            │ %s │ %s │ %s │
                                                            └────┴────┴────┘
"
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<ESC>" keymap "⎋")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f1>" keymap  "①")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f2>" keymap  "②")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f3>" keymap  "③")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f4>" keymap  "④")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f5>" keymap  "⑤")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f6>" keymap  "⑥")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f7>" keymap  "⑦")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f8>" keymap  "⑧")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f9>" keymap  "⑨")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f10>" keymap "⑩")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f11>" keymap "⑪")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f12>" keymap "⑫")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "~" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "!" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "@" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "#" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "$" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "%" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "^" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "&" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "*" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "(" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns ")" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "_" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "+" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "`" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "1" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "2" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "3" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "4" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "5" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "6" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "7" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "8" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "9" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "0" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "-" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "=" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<backspace>" keymap "⌫")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "S-<tab>" keymap "⇤")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "Q" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "W" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "E" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "R" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "T" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "Y" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "U" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "I" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "O" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "P" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "{" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "}" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "|" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<tab>" keymap "⇥")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "q" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "w" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "e" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "r" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "t" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "y" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "u" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "i" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "o" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "p" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "[" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "]" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "\\" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "A" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "S" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "D" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "F" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "G" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "H" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "J" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "K" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "L" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns ":" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "\"" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "a" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "s" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "d" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "f" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "g" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "h" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "j" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "k" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "l" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns ";" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "'" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<return>" keymap "⏎")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "Z" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "X" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "C" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "V" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "B" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "N" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "M" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns ">" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "?" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "z" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "x" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "c" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "v" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "b" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "n" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "m" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "," keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "." keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "/" keymap)
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f18>" keymap "⑱")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "SPC" keymap "␣ ")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f16>" keymap "⑯")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<f17>" keymap "⑰")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<up>" keymap "🢑")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<left>" keymap "🢐")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<right>" keymap "🢓")
   (cdr255-substitutions:get-key-from-keymap-at-2-columns "<down>" keymap "🢒")
   ))

(provide 'cdr255-substitutions)
;;; cdr255-substitutions.el ends here
