;;; cdr255-readme.el --- Yewscion's README library  -*- lexical-binding: t; -*-

;; Copyright (C) 2025 Claire Rodriguez

;; Author: Claire Rodriguez <yewscion@gmail.com>
;; Keywords: lisp
;; Version: 0.0.4

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; 

;;; Code:

(defvar cdr255-readme:%guix-shell-addons%
  `(
    ("apl" . ,(string-join (list "apl") " "))
    ("c" . ,(string-join (list "gcc-toolchain") " "))
    ("cpp" . ,(string-join (list "gcc-toolchain") " "))
    ("common-lisp" . ,(string-join (list "sbcl") " "))
    ("doc" . ,(string-join (list "texinfo") " "))
    ("emacs" . ,(string-join (list "emacs") " "))
    ("forth" . ,(string-join (list "gforth") " "))
    ("guile" . ,(string-join (list "guile" "--" "guile") " "))
    ("java" . ,(string-join (list "icedtea:jdk" "icedtea") " "))
    ("ocaml" . ,(string-join (list "ocaml" "ocaml-utop") " "))
    ("prolog" . ,(string-join (list "swi-prolog") " "))
    ("zig" . ,(string-join (list "zig") " "))
    )
  "Defines the extra packages to include when using guix shell for
various kinds of projects.

This is a DATUM.

This is an ASSOCIATION LIST (ALIST).

Fields
======

Keys <string>: The various project types expected.

Values <string>: The GNU Guix packages that need to be added to
                 guix shell in order for the testing environment
                 to be minimally useful.")

(defvar cdr255-readme:%build-systems%
  `(
    ("apl" . "gnu")
    ("c" . "gnu")
    ("cpp" . "gnu")
    ("common-lisp" . "gnu")
    ("doc" . "gnu")
    ("emacs" . "emacs")
    ("forth" . "gnu")
    ("guile" . "gnu")
    ("java" . "gnu")
    ("ocaml" . "gnu")
    ("prolog" . "gnu")
    ("zig" . "gnu")
    )
  "Defines the build-systems to use when compiling manually
various kinds of projects.

This is a DATUM.

This is an ASSOCIATION LIST (ALIST).

Fields
======

Keys <string>: The various project types expected.

Values <string>: A string which represents one of the build
                 systems used.")

(defvar cdr255-readme:%guix-preamble%
  (string-join
   (list
    "*** GNU Guix\n"
    "If You use [[https://guix.gnu.org/][GNU Guix]], this package is on"
    "[[https://sr.ht/~yewscion/yewscion-guix-channel/][my channel]].\n"
    "Once You have it set up, You can just run:\n"
    "#+begin_src bash"
    "guix pull")
   "\n")
  "The first immutable section of the GNU Guix section of the README
file.

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%guix-shell%
  (string-join
   (list
    "#+end_src\n"
    "If You just want to try it out, You can use Guix Shell instead:\n"
    "#+begin_src bash")
   "\n")
  "The immutable section describing using guix shell in the GNU Guix
section of the README file.

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%guix-cast-section%
  (string-join
   (list
    "#+end_src\n"
    "And if You'd rather just try it out without my channel, You can"
    "clone this repo and then do:\n"
    "#+begin_src bash")
   "\n")
  "The immutable section describing using the cast.sh script in the
GNU Guix section of the README file.

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%guix-postamble%
  (string-join
   (list
    "./cast.sh"
    "#+end_src\n"
    "Either of these will create a profile with *just* this project"
    "in it, to mess around with.")
   "\n")
  "The immutable section which ends the GNU Guix section of the
README file.

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%src-gnu-preamble%
  (string-join
   (list
    "*** Source\n"
    "If You don't want to use [[https://guix.gnu.org/][GNU Guix]],"
    "You can clone this repo and install it in the normal way:\n"
    "#+begin_src bash")
   "\n")
  "The first immutable section in the manual build and install
section of the README file when using the GNU build
system (autotools and/or incant.sh).

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%src-gnu-postamble%
  (string-join
   (list
    "./configure"
    "make"
    "make check"
    "make install"
    "#+end_src\n"
    "If You don't want to use git, or would rather stick with an"
    "actual release, then see the tagged releases for some tarballs"
    "of the source.\n"
    "The needed dependencies are tracked in the =DEPENDENCIES.txt= file"
    "to support this use case.\n"
    )
   "\n")
  "The last immutable section in the manual build and install
section of the README file when using the GNU build
system (autotools and/or incant.sh).

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%src-emacs-preamble%
  (string-join
   (list
    "*** Source\n"
    "If You don't want to use [[https://guix.gnu.org/][GNU Guix]],"
    "You can clone this repo and install it in the normal way"
    "(assuming You have an =~/.emacs.d/init.el=, and that"
    "=~/.emacs.d/lisp= is in Your load-path):\n"
    "#+begin_src bash")
   "\n")
  "The first immutable section in the manual build and install
section of the README file when using the Emacs build system
system (~/.emacs.d/lisp).

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%src-emacs-middle%
  (string-join
   (list
    "cp -v *.el ~/.emacs.d/lisp/"
    (concat "find ~/.emacs.d/lisp -type f -name *.el -exec \\\n"
            "emacs --batch --eval \"(byte-compile-file \\\"{}\\\")\" \\;"))
   "\n")
  "The actual install and compile section in the manual build and
install section of the README file when using the Emacs build
system system (~/.emacs.d/lisp).

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%src-emacs-postamble%
  (string-join
   (list
    "#+end_src\n"
    "Then, in Emacs, it's just a matter of restarting (or calling "
    "=M-x package-initialize= again!) and You should be good to go.\n"
    "Note that I personally prefer the below code to a simple"
    "=(add-to-list 'load-path \"~/.emacs.d/lisp/\"):\n"
    "#+begin_src emacs-lisp"
    "(let ((default-directory  \"~/.emacs.d/lisp/\"))"
    "(normal-top-level-add-subdirs-to-load-path))"
    "#+end_src\n"
    "This method of adding =~/.emacs.d/lisp/= to =load-path= allows"
    "each and every subdirectory thereof—omitting directories whose"
    "names do not start with letters or digits, those named =RCS= or"
    "=CVS=, and those containing a file named .nosearch—to be added,"
    "and therefore for the directory to be kept organized and neat."
    "You can also just open the =*.el= files and run =M-x eval-buffer=,"
    "but that only lasts for the current session.\n"
    "If You don't want to use git, or would rather stick with an"
    "actual release, then see the tagged releases for some tarballs"
    "of the source.\n"
    "You can also install using the standard autotools sequence of"
    "=./configure && make && make check && make install="
    "as I use GNU Autotools for all of my projects. The needed"
    "dependencies are tracked in the =DEPENDENCIES.txt= file"
    "to support this use case.\n")
   "\n")
  "The last immutable section in the manual build and install
section of the README file when using the Emacs build system
system (~/.emacs.d/lisp).

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%contrib-preamble%
  (string-join
   (list
    "** Contributing"
    "I prefer using the"
    "[[https://git-scm.com/docs/git-send-email][Git Send Email]] process"
    "for accepting changes to my projects. Please Development is expected"
    "to be done using [[https://guix.gnu.org/][GNU Guix]].\n"
    "If You have =guix= set up, You should be able to enter a development"
    "environment with the following:\n"
    "#+begin_src bash")
   "\n")
  "The first immutable section in the contribution section of the README
file.

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%contrib-postamble%
  (string-join
   (list
    "guix shell -D -f guix.scm bash --pure"
    "#+end_src\n"
    "If You've made changes without the above precautions, those changes will"
    "need to be confirmed to work in the above environment before merge.")
   "\n")
  "The last immutable section in the contribution section of the README
file.

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defvar cdr255-readme:%binary-message%
  (string-join
   (list
    "Any binaries or scripts will be available in Your =$PATH=. A"
    "list of these is maintained in the info file. They all also"
    "have the =--help= flag, so if You prefer learning that way,"
    "that is also available.\n")
   "\n")
  "The immutable section in the Usage section of the README
file concerning where files are generally installed.

This is a DATUM.

This is a STRING.

Fields
======

None.")

(defun cdr255-readme:make-gnu-build-instructions (project)
  "Builds the instructions for building the PROJECT using the
standard GNU Build System (Autotools / incant.sh).

This is a CALCULATION.

Arguments
=========

PROJECT <string>: The name of the project (really the git
repository directory name).

Returns
=======

A <string> representing the full instructions for building the
PROJECT using GNU Autotools / incant.sh.

Impurities
==========

Uses Global Constants; Unless rendered mutable, there are no
impurities."
  (string-join
   (list
    cdr255-readme:%src-gnu-preamble%
    (concat "git clone https://git.sr.ht/~yewscion/ " project)
    (concat "cd " project)
    cdr255-readme:%src-gnu-postamble%)
   "\n"))

(defun cdr255-readme:make-emacs-build-instructions (project)
  "Builds the instructions for building the PROJECT using the
standard Emacs Build System (~/.emacs.d/lisp).

This is a CALCULATION.

Arguments
=========

PROJECT <string>: The name of the project (really the git
repository directory name).

Returns
=======

A <string> representing the full instructions for building the
PROJECT in the standard Emacs way.

Impurities
==========

Uses Global Constants; Unless rendered mutable, there are no
impurities."
  (string-join
   (list
    cdr255-readme:%src-emacs-preamble%
    (concat "git clone https://git.sr.ht/~yewscion/ " project)
    (concat "cd " project)
    cdr255-readme:%src-emacs-middle%
    (concat "cat >> ~/.emacs.d/init.el <<< (require '" project ")")
    cdr255-readme:%src-emacs-postamble%)
   "\n"))

(defun cdr255-readme:make-guix-shell-addons (type-of-project)
  (let ((result (cdr255:string-alist-dereference
                 type-of-project
                 cdr255-readme:%guix-shell-addons%)))
    (if (not result)
        "--check"
      result)))

(defun cdr255-readme:guix-instructions (project type namespace main-lib)
  "Templating function for the 'Guix' section of my README.md files."
  (interactive)
  (string-join
   (list
    cdr255-readme:%guix-preamble%
    (concat "guix install " project)
    cdr255-readme:%guix-shell%
    (string-join (list "guix shell --pure --rebuild-cache -v4"
                       project
                       "bash"
                       (cdr255-readme:make-guix-shell-addons type))
                 " ")
    cdr255-readme:%guix-cast-section%
    (concat "cd " project)
    cdr255-readme:%guix-postamble%)
   "\n"))
(defun cdr255-readme:src-instructions (project type namespace main-lib) 
  "Templating function for the 'Source' section of my README.md files."
  (interactive)
  (let ((build-system (cdr255:string-alist-dereference
                       type cdr255-readme:%build-systems%)))
    (cond ((string= build-system "gnu")
           (cdr255-readme:make-gnu-build-instructions project))
          ((string= build-system "emacs")
           (cdr255-readme:make-emacs-build-instructions project))
          (t
           ""))))
(defun cdr255-readme:install-instructions (project
                                                  &optional type namespace main-lib)
  "Templating function for the 'Install' section of my README.md files."
  (interactive)
  (let ((type (if type type "guile"))
        (namespace (if namespace namespace "cdr255"))
        (main-lib (if main-lib main-lib "core")))
    (string-join
     (list
      "** Installation"
      (cdr255-readme:guix-instructions project type namespace main-lib)
      (cdr255-readme:src-instructions project type namespace main-lib))
     "\n")))

(defun cdr255-readme:contrib-instructions (project)
  "Templating function for the 'Contributing' section of my
README.md files."
  (interactive)
  (string-join
   (list
    cdr255-readme:%contrib-preamble%
    (concat "cd " project)
    cdr255-readme:%contrib-postamble%)
   "\n"))

(defun cdr255-readme:license-instructions (project license)
  "Templating function for the 'License' section of my README.md
files."
  (interactive)
  (string-join
   (list
    "** License"
    (concat
     "The =" project "= project and all associated files are ©2024")
    "Christopher Rodriguez, but licensed to the public at large under"
    (concat
     "the terms of the "
     (cond
      ((string= license "agpl")
       "[[https://www.gnu.org/licenses/agpl-3.0.html][GNU AGPL3.0+]]")
      ((string= license "fdl")
       "[[https://www.gnu.org/licenses/fdl-1.3.en.html][GNU FDL1.3+]]")
      (t
       "project's"))
     " license.\n")
    "Please see the =LICENSE= file and the above link for more"
    "information.")
   "\n"))

(defun cdr255-readme:std-usage-instruction-header (project)
  "Templating function for the header of the 'Usage' section of my
README.md files."
  (string-join
   (list
    "** Usage\n"
    (concat "Full usage is documented in the =doc/" project ".info=")
    "file. Here are only generic instructions.\n")
   "\n"))

(defun cdr255-readme:std-usage-instructions
    (project &optional
             type namespace main-lib
             binary-message)
  "Templating function for the 'Usage' section of my README.md files."
  (interactive)
  (let ((type (if type type "guile"))
        (namespace (if namespace namespace "cdr255"))
        (main-lib (if main-lib main-lib "core"))
        (header (cdr255-readme:std-usage-instruction-header
                 project))
        (binary-message (if binary-message binary-message
                          cdr255-readme:%binary-message%)))
    (cond ((string= type "guile")
           (concat
            header
            "You should be able to access all of\nits exported"
            " functions in guile by using its modules:\n\n"
            "#+begin_src scheme\n"
            "(use-modules ("
            namespace
            " "
            main-lib
            "))\n"
            "#+end_src\n\n"
            binary-message))
          ((string= type "emacs")
           (concat
            header
            "Once ="
            project
            "= in installed, You should be able to access all of\nits "
            "functionality in emacs. You may need to add the following"
            " to\nYour init file: \n#+begin_src lisp\n(require '"
            namespace
            ")\n"
            "#+end_src\n\n"))
          ((string= type "java")
           (concat
            header
            "You should be able to access the libaries related to this "
            "project so long as\nYour =CLASSPATH= is set properly. This "
            "can be accomplished either by setting\nthe environment "
            "variable itself, or by calling =java= et. al with the =-cp=\n"
            "flag. When possible, jars are installed alongside the "
            "classfiles themselves,\nin order to allow for the most "
            "versatility when it comes to the end-user's\nwishes.\n\n"
            binary-message))
          ((string= type "apl")
           (concat
            header
            "GNU APL libraries are interestingly different than most "
            "languages, as they\nare organized in workspaces.\n\nIn order "
            "to use the libraries included here, there are a few "
            "options\navailable:\n\n1. You can simply =)PCOPY= the =.apl= "
            "files themselves.\n\n2. Second, You can use the config file or "
            "=)LIBS n <path>= to set\none of Your nine workspaces to the "
            "directory where the files are\ninstalled.\n\n3. Finally, You "
            "can also simply copy the functions You want to use (I\nwould "
            "enjoy a commented attribution, if possible) provided You still "
            "adhere\nto the =LICENSE=.\n\n"
            binary-message))
          ((string= type "common-lisp")
           (concat
            header
            "Common lisp actually has an\n[[https://yewscion.com/common-"
            "common-lisps-in-guix.html][interesting issue]]\nright now in "
            "GNU Guix, so be aware of that if using more than one\n"
            "implementation.\n\nOther than that, loading using =asdf= is "
            "usually the way to go."
            binary-message))
          ((string= type "clojure")
           (concat
            header
            "Clojure has a few design decisions that makes working with it "
            "in GNU Guix\ndifficult, but the basics are the same as normal. "
            "Just remember: No automatic\ndependency management, and no "
            "=leiningen=."
            binary-message))
          ((string= type "c")
           (concat
            header
            "You should be able to access the libraries related to this "
            "project by\nreferencing the =/usr/lib= directory, as normal. "
            "If a subdirectory was\nneeded, I most likely chose "
            "=/usr/lib/yewscion=. If You are using GNU Guix,\nthe actual "
            "directory will be slightly different, depending on Your "
            "profile\nsetup.\n\nYou should be able to access the headers "
            "using the standard =#include=\nsyntax.\n\n"
            binary-message))
          ((string= type "other")
           (concat
            header
            binary-message)))))

(provide 'cdr255-readme)
