;;; cdr255.el --- Yewscion's Elisp Library           -*- lexical-binding: t; -*-

;; Copyright (C) 2025 Claire Rodriguez

;; Author: Claire Rodriguez <yewscion@gmail.com>
;; Keywords: lisp
;; Version: 0.0.4

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.

;;; Commentary:

;; 

;;; Code:

(require 'tmm)

(defun cdr255:selection-menu (title item-alist)
  "Show a menu to the user to select from.

This is an ACTION.

Arguments
=========

TITLE <string>: The title of the menu to display.

ITEM-ALIST <<list> of <cons>>: The items the user can choose
from, in the form:
(list (cons \"Item Name\" #'procedure-to-execute) …)

Returns
=======

<undefined>

Impurities
==========

Used solely for its side effects. I/O."
  (eval
   (let ((tmm-completion-prompt (concat title "\n\n")))
     (tmm-prompt (list ""  (cons "" item-alist) nil nil nil)))))

(defun cdr255:create-temp-file-menu-alist (string-and-file-alist)
  "Transform an alist of Names and Filenames into an alist suitable
for cdr:selection-menu, with the goal of opening temporary
buffers containing the contents of the files specified.

This is a CALCULATION.

Arguments
=========

STRING-AND-FILE-ALIST <<list> of <lists> of <strings>>: A list in
the form:

'((\"Title\" \"/foo/bar/baz\") …)

Returns
=======

A <<list> of <cons>> which is suitable for cdr:selection-menu,
specifying the items the user can choose from, in the form:
(list (cons \"Item Name\" #'procedure-to-execute) …)

The #'procedure-to-execute will open a new temporary buffer with
\"Item Name\" as a prefix, and the contents of the associated
file inserted.

Impurities
==========

None."
  (mapcar 
   (lambda (x)
     (cons (car x) `(cdr255:temp-buffer-with-file-contents ,(car x) ,(cadr x))))
   string-and-file-alist))

(defun cdr255:create-comment-insertion-menu-alist (string-and-file-alist)
  "Transform an alist of Names and Filenames into an alist suitable
for cdr:selection-menu, with the goal of opening temporary
buffers containing the contents of the files specified.

This is a CALCULATION.

Arguments
=========

STRING-AND-FILE-ALIST <<list> of <lists> of <strings>>: A list in
the form:

'((\"Title\" \"/foo/bar/baz\") …)

Returns
=======

A <<list> of <cons>> which is suitable for cdr:selection-menu,
specifying the items the user can choose from, in the form:
(list (cons \"Item Name\" #'procedure-to-execute) …)

The #'procedure-to-execute will open a new temporary buffer with
\"Item Name\" as a prefix, and the contents of the associated
file inserted.

Impurities
==========

None."
  (mapcar 
   (lambda (x)
     (cons (car x) `(insert
                     (cdr255:comment-multiline-string (f-read-text ,(cadr x))))))
   string-and-file-alist))

(defun cdr255:comment-multiline-string (string)
  "Comments STRING using the current mode's comment syntax, taking into
account that newlines will need a new comment character.

This is an ACTION.

Arguments
=========

STRING <string>: The string to turn into a comment.

Returns
=======

A <string> representing the original STRING with comment padding
and characters prepended to each line.

Impurities
==========

Relies on current buffer's major mode for comment characters."
  (concat comment-start
          comment-padding
          (string-replace "\n"
                          (concat "\n"
                                  comment-start
                                  comment-padding)
                          string)))

(defun cdr255:temp-buffer-display-string (title content)
  "Create a new temporary buffer starting with TITLE and put the
CONTENT string inside of it.

This is an ACTION.

Arguments
=========

TITLE <string>: The prefix for the new temporary buffer.

CONTENT <string>: The data to insert into the temporary buffer.

Returns
=======

<undefined>

Impurities
==========

Used solely for its side effects. I/O."
  (let ((temp-buffer-name (make-temp-name (concat "[" title "] "))))
    (generate-new-buffer temp-buffer-name)
    (set-buffer temp-buffer-name)
    (insert content)
    (local-set-key "q" 'kill-current-buffer)
    (read-only-mode nil)
    (switch-to-buffer temp-buffer-name)
    (message (concat "Opening " title "…"))))

(defun cdr255:temp-buffer-with-file-contents (title filename)
  "Create a new temporary buffer starting with TITLE and put the contents
of FILENAME inside of it.

This is an ACTION.

Arguments
=========

TITLE <string>: The prefix for the new temporary buffer.

FILENAME <string>: The file from which to insert data.

Returns
=======

<undefined>

Impurities
==========

Used solely for its side effects. Relies on current system state. I/O."
  (cdr255:temp-buffer-display-string title (f-read-text filename)))

(defun cdr255:temp-buffer-with-command-output (title command)
  "Create a new temporary buffer starting with TITLE and put the
output of COMMAND inside of it.

This is an ACTION.

Arguments
=========

TITLE <string>: The prefix for the new temporary buffer.

COMMAND <string>: The shell command to run for its output.

Returns
=======

<undefined>

Impurities
==========

Used solely for its side effects. Relies on current system state. I/O."
  (cdr255:temp-buffer-display-string title
                                     (substring-no-properties (shell-command-to-string command))))

(defun cdr255:empty-string-to-nil (supplied-string)
  "Return NIL if SUPPLIED-STRING is the empty string \"\". Otherwise,
returns SUPPLIED-STRING.

This is a CALCULATION.

Arguments
=========

SUPPLIED-STRING <string>: The string to check for the empty string.

Returns
=======

Either NIL or the SUPPLIED-STRING.

Impurities
==========
None."
  (if (string= supplied-string "")
      nil
    supplied-string))

(defun cdr255:toggle-proc-var-alist (proc var alist)
  "Uses PROC to set VAR to the value of VAR in ALIST.

This is an ACTION.

Arguments
=========

PROC <procedure>: The procedure to run on the value associated with VAR in ALIST.

VAR <key>: The key to look up in ALIST. Often also mutated by PROC.

ALIST <association list>: A list of pairs of items, where VAR is
a key, and the value stored will be fed to PROC.

Returns
=======
<undefined>

Impurities
==========
Used entirely for its side effects. Relies on current system state."
  (apply proc (list (cadr (assoc var alist)))))

(defun cdr255:delete-previous-line ()
  "Delete the previous line in the current buffer.

This is an ACTION.

Arguments
=========

None.

Returns
=======

<undefined>.

Impurities
==========

Used entirely for its side effects."
  (previous-line)
  (cdr:delete-current-line))

(defun cdr255:delete-current-line ()
  "Delete the current line in the current buffer.

This is an ACTION.

Arguments
=========

None.

Returns
=======

<undefined>.

Impurities
==========

Used entirely for its side effects."
  (beginning-of-line)
  (kill-line)
  (delete-char 1))

(defun cdr255:insert-contents-of-url (url)
  "Insert the content from URL at the current point.

This is an ACTION.

Arguments
=========

URL <string>: The URL from which to retrieve data.

Returns
=======

<undefined>.

Impurities
=========="

  (interactive "sURL? ")
  (insert (cdr255:get-url-contents-as-string url)))

(defun cdr255:get-url-contents-as-string (url)
  "Get the content from URL as a string.

This is an ACTION.

Arguments
=========

URL <string>: The URL from which to retrieve data.

Returns
=======

A <string> representing the data retrieved from URL.

Impurities
==========

Network IO."
  (with-current-buffer (url-retrieve-synchronously
                        url)
    (buffer-string)))

(defun cdr255:buffer-as-string ()
  "Get the entire contents of the current buffer as a string.

This is an ACTION.

Arguments
=========

None.

Returns
=======

A <string> representing the entire contents of the current
buffer, with no properties or extraneous information.

Impurities
==========

Relies on the current buffer state."
  (buffer-substring-no-properties (point-min) (point-max)))

(defun cdr255:string-alist-dereference (key-string string-alist)
  "Obtain the value stored in STRING-ALIST by using the KEY-STRING
as a key.

This is a COMPUTATION.

Arguments
=========

KEY-STRING <string>: A string to use as the key for an assoc call
across STRING-ALIST.

STRING-ALIST <association-list>: The store of data we are
dereferencing from using the KEY-STRING. Assumed to be an alist
with members like (<string> <object>).

Returns
=======

An <object> representing the value stored in STRING-ALIST under
the key KEY-STRING. Will be <nil> if KEY-STRING is not found.

Impurities
==========

None."
  (alist-get key-string string-alist nil nil #'string=))

(defun cdr255:line-banner-calculation (phrase line-width filler-character)
  "Build a string containing a specific PHRASE, surrounded by
FILLER-CHARACTERs, and padded with two spaces to fit inside of
LINE-WIDTH. Originally for use in comments.

This is a CALCULATION.

Arguments
=========

PHRASE <string>: The phrase to be emphasized with the banner.

LINE-WIDTH <number>: The desired total width of the string.

FILLER-CHARACTER <character>: The character to use to denote the
                              banner, and to pad the banner so
                              that the PHRASE is reasonably
                              centered in the LINE-WIDTH.

Returns
=======

A <string> which represents the PHRASE enclosed by
FILLER-CHARACTERs, and padded with two spaces. If the phrase is
too long to fit with the surrounding characters in the desired
line-width, the phrase is truncated and an elipsis added to
denote this.

Impurities
==========

None."

  (cond ((>= (length phrase) (- line-width 5))
         (string-join
          (list
           (string filler-character)
           (concat (substring phrase 0 (- line-width 7)) "…")
           (string filler-character))
          "  "))
        ((< (length phrase) (- line-width 5))
         (let* ((non-phrase-width (- line-width (+ (length phrase) 4)))
                (prefix-width (if (= (mod non-phrase-width 2) 1)
                                  (+ 1 (/ non-phrase-width 2))
                                (/ non-phrase-width 2)))
                (postfix-width (/ non-phrase-width 2)))
           (string-join
            (list
             (make-string prefix-width filler-character)
             phrase
             (make-string postfix-width filler-character))
            "  ")))))

(defun cdr255:line-comment (phrase &optional line-width filler-character)
  "Build a string containing a specific PHRASE, surrounded by
FILLER-CHARACTERs, and padded with two spaces to fit inside of
LINE-WIDTH. Originally for use in comments.

This is an ACTION.

This wraps the CALCULATION cdr255:line-banner-calculation with
defaults.

Arguments
=========

PHRASE <string>: The phrase to be emphasized with the banner.

LINE-WIDTH <number>: The desired total width of the
                     string. Defaults to the current value of
                     'fill-column.

FILLER-CHARACTER <character>: The character to use to denote the
                              banner, and to pad the banner so
                              that the PHRASE is reasonably
                              centered in the
                              LINE-WIDTH. Defaults to the current
                              value of 'comment-start.

Returns
=======

A <string> which represents the PHRASE enclosed by
FILLER-CHARACTERs, and padded with two spaces. If the phrase is
too long to fit with the surrounding characters in the desired
line-width, the phrase is truncated and an elipsis added to
denote this.

Impurities
==========

Default values use global variables."
  (unless line-width (setq line-width fill-column))
  (unless filler-character (setq filler-character
                                 (string-to-char comment-start)))
  (cdr255:line-banner-calculation phrase line-width filler-character))

(defun cdr255:insert-line-banner-comment (phrase)
  "Inserts a line before the current point that emphasized PHRASE
within a comment.

This is an ACTION.

Arguments
=========

PHRASE <string>: The phrase to be emphasized with the banner.

Returns
=======

<undefined>.

Impurities
==========

Modifies buffer state. Relies on global (buffer-local) variables."
  (interactive "*sPhrase for Banner: ")
  (let ((full-line (make-string fill-column (string-to-char comment-start))))
    (save-mark-and-excursion
      (insert full-line)
      (newline)
      (insert (cdr255:line-comment phrase nil nil))
      (newline)
      (insert full-line)
      (newline))))


(defun cdr255:eval-string (item)
  "Evaluates the string ITEM as Emacs Lisp code.

This is an ACTION.

Arguments
=========

ITEM <string>: A string of valid Emacs Lisp code, which will be
evaluated. THIS IS NOT SAFE, AS IT ALLOWS ARBITRARY CODE
EXECUTION.

Returns
=======

<undefined>, as this procedure will return the return value of
evaluating the string ITEM as Emacs Lisp code. Could be anything
or nothing.


Impurities
==========

Evaluates arbitrary code."
    (eval (car (read-from-string (format "(progn %s)" item)))))


(defun cdr255:ensure-directory-has-trailing-slash (directory-string)
"A guard to ensure that the given DIRECTORY-STRING has a trailing
forward slash at the end.

This is a CALCULATION.

Arguments
=========

DIRECTORY-STRING <string>: The string in question.

Returns
=======

A <string> represeting the original DIRECTORY-STRING, but with an
added slash if it did not originally end with one.

Impurities
==========

None."
  (if (string= (substring-no-properties directory-string -1) "/")
                            directory-string
    (concat directory-string "/")))

(defun cdr255:cleanup-string-for-filename (x)
  "Cleanup X so that it doesn't contain difficult characters if used as a file name.

This is a CALCULATION.

Arguments
=========

X <string>: The original string, which will be modified by this procedure.

Returns
=======

A <string> representing the original string with a set of
replacement operations applied to make it easier to use as a file
name.

Impurities
==========

None."
  ;; Remove dashes and underscores at the beginning or end of the string
  (replace-regexp-in-string
   "\\(^[-_]+\\|[-_]+$\\)" ""
   ;; Remove multiple dashes
   (replace-regexp-in-string
    "-+" "-"
    ;; Remove multiple underscores
    (replace-regexp-in-string
     "_+" "_"
     ;; Replace spaces with dashes
     (replace-regexp-in-string
      "[ ]" "-"
      ;; Replace common punctuation with underscores
      (replace-regexp-in-string
       "[+(){}:;\".,?/<>!#$&*]" "_"
       ;; Remove all single quotes
       (replace-regexp-in-string
        "[']" ""
        ;; Downcase the original string
        (downcase x))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;  2024-08-19  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255:get-date-as-string (&optional date)
  (let ((time (decode-time date)))
     (format "%04d-%02d-%02d"
             (nth 5 time)
             (nth 4 time)
             (nth 3 time))))

(defun cdr255:first-line-and-name-of-file (file)
  (cons
   (cdr255:first-line-of-file file)
   file))

(defun cdr255:first-line-of-file (file)
  (car
   (s-lines
    (f-read file))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;  [2024-09-16 Mon 10:32]  ;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun cdr255:eval-string (item)
  "Evaluates ITEM as a piece of Emacs Lisp code."
    (eval (car (read-from-string (format "(progn %s)" item)))))

(defun cdr255:get-prior-sexp-coords ()
  "Get the nearest prior S-expression's coordinates as a cons cell.

This is an ACTION.

Arguments
=========

None.

Returns
=======

A <cons cell> like <<char-number><char-number>>, with the first
value being the start of the prior S-expression, and the last
value being the end of the prior S-expression.

Impurities
==========

Relies on the current buffer state."
  (save-mark-and-excursion
    (let ((end (point)))
      (thing-at-point--beginning-of-sexp)
      (cons (point) end))))

(defun cdr255:get-prior-sexp-as-string ()
  "Get the nearest prior S-expression as a string.

This is an ACTION.

Arguments
=========

None.

Returns
=======

A <string> representing the nearest prior S-expression.

Impurities
==========

Relies on the current buffer state."
  (let ((coords (cdr255:get-prior-sexp-coords)))
    (buffer-substring-no-properties (car coords) (cdr coords))))

(defun cdr255:format-princ (value)
  "Generate a string representing value as a human-readable string.

This is a CALCULATION.

Arguments
=========

VALUE <object>: This is what we are generating a string to represent.

Returns
=======

A <string> representing the VALUE as a human-readable string.

Impurities
==========

None."
  (format "%s" value))
(defun cdr255:format-prin1 (value)
  "Generate a string representing value as an S-expression.

This is a CALCULATION.

Arguments
=========

VALUE <object>: This is what we are generating a string to represent.

Returns
=======

A <string> representing the VALUE as an S-expression.

Impurities
==========

None."  
  (format "%S" value))
(defun cdr255:format-2-digit (value)
  "Generate a string representing value as a 2 digit integer.

This is a CALCULATION.

Arguments
=========

VALUE <object>: This is what we are generating a string to represent.

Returns
=======

A <string> representing the VALUE as a 2 digit integer.

Impurities
==========

None."  
  (format "%02d" value))
(defun cdr255:format-2-mantissa (value)
  "Generate a string representing value as a number in base 10 with a mantissa of length 2.

This is a CALCULATION.

Arguments
=========

VALUE <object>: This is what we are generating a string to represent.

Returns
=======

A <string> representing the VALUE as a number in base 10 with a mantissa of length 2.

Impurities
==========

None."  
  (format "%0.2f" value))

(defun cdr255:eval-sexp-string-to-proper-string (sexp-string)
  "Generates a string representing the result of evaluating
SEXP-STRING, with some opinionated ways of representing various
types of values.

This is a CALCULATION.

Arguments
=========

SEXP-STRING <string>: A representation of a valid S-expression
that we will be evaluating a using as a basis for our generated
string.

Returns
=======

A <string> representing the result of evaluating SEXP-STRING.

Impurities
==========

None, assuming the S-Expression supplied as SEXP-STRING is pure."
  (let ((result (cdr255:eval-string sexp-string)))
    (cond
     ((stringp result) (cdr255:format-princ result))
     ((= (round result) result) (cdr255:format-2-digit result))
     ((numberp result) (cdr255:format-2-mantissa result))
     (t (cdr255:format-prin1 result)))))

(defun cdr255:eval-prior-sexp-to-string ()
  "Returns the result of evaluating the nearest prior S-expression as a string.

This is an ACTION.

Arguments
=========

None.

Returns
=======

A <string> representing the result of evaluating the nearest prior S-expression.

Impurities
==========

Relies on current buffer state."
  (cdr255:eval-sexp-string-to-proper-string (cdr255:get-prior-sexp-as-string)))

(defun cdr255:eval-prior-sexp-message ()
  "Displays the result of evaluating the nearest prior S-expression as a message.

This is an ACTION.

Arguments
=========

None.

Returns
=======

<undefined>

Impurities
==========

Relies on current buffer state, undefined return value."
  (interactive)
  (message (cdr255:eval-prior-sexp-to-string)))

(defun cdr255:eval-prior-sexp-insert ()
  "Inserts the result of evaluating the nearest prior S-expression at point.

This is an ACTION.

Arguments
=========

None.

Returns
=======

<undefined>

Impurities
==========

Relies on and modifies current buffer state, undefined return value."
  (interactive)
  (insert (cdr255:eval-prior-sexp-to-string)))

(defun cdr255:eval-prior-sexp-replace ()
  "Inserts the result of evaluating the nearest prior S-expression at point, removing the S-expression.

This is an ACTION.

Arguments
=========

None.

Returns
=======

<undefined>

Impurities
==========

Relies on and modifies current buffer state, undefined return value."
  (interactive)
  (let ((coords (cdr255:get-prior-sexp-coords)))
    (insert (cdr255:eval-prior-sexp-to-string))
    (delete-region (car coords) (cdr coords))))

(defun cdr255:copy-cleaned-up-string (&optional str)
  "Transliterates a `dirty' string, STR, into one which has been cleaned up.

This is an ACTION.

Arguments
=========

STR<string>: An optional string which will be used (if it is supplied)
as the 'dirty' string. If `NIL', then the region is used as a string instead.

Returns
=======

A version of the `dirty' string with a bunch of the difficult-to-parse
characters replaced, and transformed to all lower case.

Impurities
==========

Can rely on current region. If optional STR argument is specified, this is a CALCULATION."

  (interactive)
  (let ((str (if str str (buffer-substring-no-properties (region-beginning) (region-end)))))
    (kill-new (downcase (replace-regexp-in-string "[!?:;\",.<>()+=@#$%^&*~`|\\]" "_"
                              (replace-regexp-in-string "[ 	]" "-"
                                                        (replace-regexp-in-string
                                                         "[_']" "" str)))))))

(provide 'cdr255)
;;; cdr255.el ends here




