;;; sxml-mode.el --- A mode for editing SXML files   -*- lexical-binding: t; -*-

;; Copyright (C) 2024  Claire Rodriguez

;; Author: Claire Rodriguez
;; Version: 0.0.1
;; Keywords: lisp

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <https://www.gnu.org/licenses/>.


;;; Commentary:

;; 

;;; Code:

(require 'cdr255)

;;;###autoload
(defgroup sxml-mode
  nil
  "Options for the use of SXML Mode."
  :prefix "sxml-")

;;;###autoload
(defcustom
  sxml-template-alist
  '((new-xhtml5-page .
              "; Variables: Title ; Short Description ; Introduction ; section-1 ;
;            Section 1; Section 1 Description ; Section 1 Content
`(body
  (header
   (header-group
    (h1 \"Title\")
    (p \"Short Description\"))
   (p \"Introduction\"))
  (hr)
  (section (@ (id \"section-1\"))
           (header-group
            (h2 \"Section 1\")
            (p \"Section 1 Description\"))
           (p \"Section 1 Content\"))
  (hr)
  (footer
   (small
    (p \"All original content on this website is ©2024 Claire Rodriguez, but
licensed to the public at large under the following license:\"
       ,%cc-by-sa-4.0
       \".\"))))")
    (section . "; Variables: section-1 ; Section 1; Section 1 Description ; Section 1 Content
(section (@ (id \"section-1\"))
         (header-group
          (h2 \"Section 1\")
          (p \"Section 1 Description\"))
         (p \"Section 1 Content\"))")
    (checkbox-dl-section . "; Variables: template ; Template ; Description
(section (@ (id \"template\"))
         (form
          (header-group
           (h2 \"Template\")
           (small
            (p \"Description\"))
           (input (@ (type \"reset\") (value \"Reset this section\"))))
          (dl
           (dt
            (input (@ (type \"checkbox\") (id \"template-1\") (name \"template-1\")))
            ,(anchor \"#\" \"A\"))
           (dd
            \"A\"))))"))
  "An alist of templates defined by their KEYS. If You add more,
 You can insert them using (sxml-insert-template 'key)."
  :group 'sxml-mode
  :type '(alist :key-type symbol :value-type string)
  :tag "SXML Templates")

;;;###autoload
(defcustom sxml-fill-column
  70
  "The longest line allowed in an SXML file."
  :group 'sxml-mode
  :type 'integer
  :tag "SXML Fill Column")

;;;###autoload
(defcustom sxml-comment-column
  16
  "Where comments inside of SXML documents should start."
  :group 'sxml-mode
  :type 'integer
  :tag "SXML Comment Start Column")

;;;###autoload
(define-derived-mode sxml-mode
  scheme-mode "SXML" "A mode for editing SXML files."
  (:Group "SXML Mode")
  (setq-local fill-column sxml-fill-column
              comment-column sxml-comment-column)
  (define-key sxml-mode-map (kbd "H-n")
            #'(lambda () (interactive) (sxml-insert-template 'new-xhtml5-page)))
  (define-key sxml-mode-map (kbd "H-s")
            #'(lambda () (interactive) (sxml-insert-template 'section)))
  (define-key sxml-mode-map (kbd "H-d")
            #'(lambda () (interactive) (sxml-insert-template 'checkbox-dl-section))))

;;;###autoload
(defun sxml-template-insert-xhtml5-page ()
  (interactive)
  (insert ))

;;;###autoload
(defun sxml-get-template (sxml-template-alist template-key)
  "Insert an SXML template from SXML-TEMPLATE-ALIST, identified by
TEMPLATE-KEY.

This is a CALCULATION.

Arguments
=========

Returns
=======

Impurities
=========="
  (interactive)
  (alist-get template-key sxml-template-alist))

;;;###autoload
(defun sxml-insert-template (template-key)
  "Insert the SXML template stored in the customizable variable
SXML-TEMPLATE-ALIST identified by TEMPLATE-KEY into the current
buffer at point.

This is an ACTION.

Arguments
=========

TEMPLATE-KEY <symbol>: The key to use in reference to the
customizable variable SXML-TEMPLATE-ALIST.

Returns
=======

<undefined>

Impurities
==========

Uses global variable (SXML-TEMPLATE-ALIST), modifies current
state of buffer."
  (save-mark-and-excursion
    (insert
     (sxml-get-template sxml-template-alist template-key))))

;;;###autoload
(add-to-list 'auto-mode-alist '("\\.sxml\\'" . sxml-mode))
;;;###autoload
(add-to-list 'interpreter-mode-alist '("sxml-tool" . sxml-mode))

(provide 'sxml-mode)
;;; sxml-mode.el ends here
